<?php

namespace TablePress\PhpOffice\PhpSpreadsheet\Calculation\Statistical\Distributions;

use TablePress\PhpOffice\PhpSpreadsheet\Calculation\ArrayEnabled;
use TablePress\PhpOffice\PhpSpreadsheet\Calculation\Exception;
use TablePress\PhpOffice\PhpSpreadsheet\Calculation\Information\ExcelError;

class LogNormal
{
	use ArrayEnabled;

	/**
	 * LOGNORMDIST.
	 *
	 * Returns the cumulative lognormal distribution of x, where ln(x) is normally distributed
	 * with parameters mean and standard_dev.
	 *
	 * @param mixed $value Float value for which we want the probability
	 *                      Or can be an array of values
	 * @param mixed $mean Mean value as a float
	 *                      Or can be an array of values
	 * @param mixed $stdDev Standard Deviation as a float
	 *                      Or can be an array of values
	 *
	 * @return array|float|string The result, or a string containing an error
	 *         If an array of numbers is passed as an argument, then the returned result will also be an array
	 *            with the same dimensions
	 */
	public static function cumulative($value, $mean, $stdDev)
	{
		if (is_array($value) || is_array($mean) || is_array($stdDev)) {
			return self::evaluateArrayArguments([self::class, __FUNCTION__], $value, $mean, $stdDev);
		}

		try {
			$value = DistributionValidations::validateFloat($value);
			$mean = DistributionValidations::validateFloat($mean);
			$stdDev = DistributionValidations::validateFloat($stdDev);
		} catch (Exception $e) {
			return $e->getMessage();
		}

		if (($value <= 0) || ($stdDev <= 0)) {
			return ExcelError::NAN();
		}

		return StandardNormal::cumulative((log($value) - $mean) / $stdDev);
	}

	/**
	 * LOGNORM.DIST.
	 *
	 * Returns the lognormal distribution of x, where ln(x) is normally distributed
	 * with parameters mean and standard_dev.
	 *
	 * @param mixed $value Float value for which we want the probability
	 *                      Or can be an array of values
	 * @param mixed $mean Mean value as a float
	 *                      Or can be an array of values
	 * @param mixed $stdDev Standard Deviation as a float
	 *                      Or can be an array of values
	 * @param mixed $cumulative Boolean value indicating if we want the cdf (true) or the pdf (false)
	 *                      Or can be an array of values
	 *
	 * @return array|float|string The result, or a string containing an error
	 *         If an array of numbers is passed as an argument, then the returned result will also be an array
	 *            with the same dimensions
	 */
	public static function distribution($value, $mean, $stdDev, $cumulative = false)
	{
		if (is_array($value) || is_array($mean) || is_array($stdDev) || is_array($cumulative)) {
			return self::evaluateArrayArguments([self::class, __FUNCTION__], $value, $mean, $stdDev, $cumulative);
		}

		try {
			$value = DistributionValidations::validateFloat($value);
			$mean = DistributionValidations::validateFloat($mean);
			$stdDev = DistributionValidations::validateFloat($stdDev);
			$cumulative = DistributionValidations::validateBool($cumulative);
		} catch (Exception $e) {
			return $e->getMessage();
		}

		if (($value <= 0) || ($stdDev <= 0)) {
			return ExcelError::NAN();
		}

		if ($cumulative === true) {
			return StandardNormal::distribution((log($value) - $mean) / $stdDev, true);
		}

		return (1 / (sqrt(2 * M_PI) * $stdDev * $value)) *
			exp(0 - ((log($value) - $mean) ** 2 / (2 * $stdDev ** 2)));
	}

	/**
	 * LOGINV.
	 *
	 * Returns the inverse of the lognormal cumulative distribution
	 *
	 * @param mixed $probability Float probability for which we want the value
	 *                      Or can be an array of values
	 * @param mixed $mean Mean Value as a float
	 *                      Or can be an array of values
	 * @param mixed $stdDev Standard Deviation as a float
	 *                      Or can be an array of values
	 *
	 * @return array|float|string The result, or a string containing an error
	 *         If an array of numbers is passed as an argument, then the returned result will also be an array
	 *            with the same dimensions
	 *
	 * @TODO    Try implementing P J Acklam's refinement algorithm for greater
	 *            accuracy if I can get my head round the mathematics
	 *            (as described at) http://home.online.no/~pjacklam/notes/invnorm/
	 */
	public static function inverse($probability, $mean, $stdDev)
	{
		if (is_array($probability) || is_array($mean) || is_array($stdDev)) {
			return self::evaluateArrayArguments([self::class, __FUNCTION__], $probability, $mean, $stdDev);
		}

		try {
			$probability = DistributionValidations::validateProbability($probability);
			$mean = DistributionValidations::validateFloat($mean);
			$stdDev = DistributionValidations::validateFloat($stdDev);
		} catch (Exception $e) {
			return $e->getMessage();
		}

		if ($stdDev <= 0) {
			return ExcelError::NAN();
		}
		/** @var float */
		$inverse = StandardNormal::inverse($probability);

		return exp($mean + $stdDev * $inverse);
	}
}
